"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClientUser = exports.ActivityPartyPrivacy = exports.ActivitySupportedPlatform = void 0;
const VoiceSettings_1 = require("./VoiceSettings");
const Channel_1 = require("./Channel");
const Guild_1 = require("./Guild");
const User_1 = require("./User");
var ActivitySupportedPlatform;
(function (ActivitySupportedPlatform) {
    ActivitySupportedPlatform["IOS"] = "ios";
    ActivitySupportedPlatform["ANDROID"] = "android";
    ActivitySupportedPlatform["WEB"] = "web";
})(ActivitySupportedPlatform || (exports.ActivitySupportedPlatform = ActivitySupportedPlatform = {}));
var ActivityPartyPrivacy;
(function (ActivityPartyPrivacy) {
    ActivityPartyPrivacy[ActivityPartyPrivacy["PRIVATE"] = 0] = "PRIVATE";
    ActivityPartyPrivacy[ActivityPartyPrivacy["PUBLIC"] = 1] = "PUBLIC";
})(ActivityPartyPrivacy || (exports.ActivityPartyPrivacy = ActivityPartyPrivacy = {}));
class ClientUser extends User_1.User {
    // #region Helper function
    async fetchUser(userId) {
        return new User_1.User(this.client, (await this.client.request("GET_USER", { id: userId })).data);
    }
    /**
     * Used to get a guild the client is in.
     *
     * @param guildId - id of the guild to get
     * @param timeout - asynchronously get guild with time to wait before timing out
     * @returns partial guild
     */
    async fetchGuild(guildId, timeout) {
        return new Guild_1.Guild(this.client, (await this.client.request("GET_GUILD", { guild_id: guildId, timeout })).data);
    }
    /**
     * Used to get a list of guilds the client is in.
     * @returns the guilds the user is in
     */
    async fetchGuilds() {
        return (await this.client.request("GET_GUILDS")).data.guilds.map((guildData) => new Guild_1.Guild(this.client, guildData));
    }
    /**
     * Used to get a channel the client is in.
     * @param channelId - id of the channel to get
     * @returns partial channel
     */
    async fetchChannel(channelId) {
        return new Channel_1.Channel(this.client, (await this.client.request("GET_CHANNEL", { channel_id: channelId })).data);
    }
    /**
     * Used to get a guild's channels the client is in.
     * @param guildId - id of the guild to get channels for
     * @returns guild channels the user is in
     */
    async fetchChannels(guildId) {
        return (await this.client.request("GET_CHANNELS", { guild_id: guildId })).data.channels.map((channelData) => new Channel_1.Channel(this.client, channelData));
    }
    /**
     * Used to get the client's current voice channel. There are no arguments for this command. Returns the [Get Channel](https://discord.com/developers/docs/topics/rpc#getchannel) response, or `null` if none.
     * @returns the client's current voice channel, `null` if none
     */
    async getSelectedVoiceChannel() {
        const response = await this.client.request("GET_SELECTED_VOICE_CHANNEL");
        return response.data !== null ? new Channel_1.Channel(this.client, response.data) : null;
    }
    /**
     * Used to join voice channels, group dms, or dms. Returns the [Get Channel](https://discord.com/developers/docs/topics/rpc#getchannel) response, `null` if none.
     * @param channelId - channel id to join
     * @param timeout - asynchronously join channel with time to wait before timing out
     * @param force - forces a user to join a voice channel
     * @returns the channel that the user joined, `null` if none
     */
    async selectVoiceChannel(channelId, timeout, force, navigate) {
        return new Channel_1.Channel(this.client, (await this.client.request("SELECT_VOICE_CHANNEL", {
            channel_id: channelId,
            timeout,
            force,
            navigate
        })).data);
    }
    /**
     * Used to leave voice channels, group dms, or dms
     * @param timeout - asynchronously join channel with time to wait before timing out
     * @param force - forces a user to join a voice channel
     */
    async leaveVoiceChannel(timeout, force) {
        await this.client.request("SELECT_VOICE_CHANNEL", {
            channel_id: null,
            timeout,
            force
        });
    }
    /**
     * Used to get current client's voice settings
     * @returns the voice setting
     */
    async getVoiceSettings() {
        return new VoiceSettings_1.VoiceSettings(this.client, (await this.client.request("GET_VOICE_SETTINGS")).data);
    }
    /**
     * Used by hardware manufacturers to send information about the current state of their certified devices that are connected to Discord.
     * @param devices - a list of devices for your manufacturer, in order of priority
     * @returns
     */
    async setCeritfiedDevices(devices) {
        await this.client.request("SET_CERTIFIED_DEVICES", { devices });
    }
    /**
     * Used to accept an Ask to Join request.
     * @param userId - the id of the requesting user
     */
    async sendJoinInvite(userId) {
        await this.client.request("SEND_ACTIVITY_JOIN_INVITE", { user_id: userId });
    }
    /**
     * Used to reject an Ask to Join request.
     * @param userId - the id of the requesting user
     */
    async closeJoinRequest(userId) {
        await this.client.request("CLOSE_ACTIVITY_JOIN_REQUEST", { user_id: userId });
    }
    /**
     * Used to join text channels, group dms, or dms. Returns the [Get Channel](https://discord.com/developers/docs/topics/rpc#getchannel) response, or `null` if none.
     * @param channelId - channel id to join
     * @param timeout - asynchronously join channel with time to wait before timing out
     * @returns the text channel that user joined
     */
    async selectTextChannel(channelId, timeout) {
        return new Channel_1.Channel(this.client, (await this.client.request("SELECT_TEXT_CHANNEL", { channel_id: channelId, timeout })).data);
    }
    /**
     * Used to leave text channels, group dms, or dms.
     * @param timeout - asynchronously join channel with time to wait before timing out
     */
    async leaveTextChannel(timeout) {
        await this.client.request("SELECT_TEXT_CHANNEL", { channel_id: null, timeout });
    }
    async getRelationships() {
        return (await this.client.request("GET_RELATIONSHIPS")).data.relationships.map((data) => {
            return new User_1.User(this.client, { ...data.user, presence: data.presence });
        });
    }
    /**
     * Used to update a user's Rich Presence.
     *
     * @param activity - the rich presence to assign to the user
     * @param pid - the application's process id
     * @returns The activity that have been set
     */
    async setActivity(activity, pid) {
        const formattedAcitivity = {
            ...activity,
            assets: {},
            timestamps: {},
            party: {},
            secrets: {}
        };
        if (activity.startTimestamp instanceof Date) {
            formattedAcitivity.timestamps.start = Math.round(activity.startTimestamp.getTime());
        }
        else if (typeof activity.startTimestamp === "number") {
            formattedAcitivity.timestamps.start = activity.startTimestamp;
        }
        if (activity.endTimestamp instanceof Date) {
            formattedAcitivity.timestamps.end = Math.round(activity.endTimestamp.getTime());
        }
        else if (typeof activity.endTimestamp === "number") {
            formattedAcitivity.timestamps.end = activity.endTimestamp;
        }
        if (activity.largeImageKey)
            formattedAcitivity.assets.large_image = activity.largeImageKey;
        if (activity.smallImageKey)
            formattedAcitivity.assets.small_image = activity.smallImageKey;
        if (activity.largeImageText)
            formattedAcitivity.assets.large_text = activity.largeImageText;
        if (activity.smallImageText)
            formattedAcitivity.assets.small_text = activity.smallImageText;
        if (activity.partyId)
            formattedAcitivity.party.id = activity.partyId;
        if (activity.partyPrivacy)
            formattedAcitivity.party.privacy = activity.partyPrivacy;
        if (activity.partySize && activity.partyMax)
            formattedAcitivity.party.size = [activity.partySize, activity.partyMax];
        if (activity.joinSecret)
            formattedAcitivity.secrets.join = activity.joinSecret;
        if (activity.spectateSecret)
            formattedAcitivity.secrets.spectate = activity.spectateSecret;
        if (activity.matchSecret)
            formattedAcitivity.secrets.match = activity.matchSecret;
        if (activity.supportedPlatforms)
            formattedAcitivity.supported_platforms = activity.supportedPlatforms;
        if (Object.keys(formattedAcitivity.assets).length === 0)
            delete formattedAcitivity["assets"];
        if (Object.keys(formattedAcitivity.timestamps).length === 0)
            delete formattedAcitivity["timestamps"];
        if (Object.keys(formattedAcitivity.party).length === 0)
            delete formattedAcitivity["party"];
        if (Object.keys(formattedAcitivity.secrets).length === 0)
            delete formattedAcitivity["secrets"];
        formattedAcitivity.instance = !!activity.instance;
        // Clean-up
        delete formattedAcitivity["startTimestamp"];
        delete formattedAcitivity["endTimestamp"];
        delete formattedAcitivity["largeImageKey"];
        delete formattedAcitivity["smallImageKey"];
        delete formattedAcitivity["largeImageText"];
        delete formattedAcitivity["smallImageText"];
        delete formattedAcitivity["partyId"];
        delete formattedAcitivity["partyPrivacy"];
        delete formattedAcitivity["partySize"];
        delete formattedAcitivity["partyMax"];
        delete formattedAcitivity["joinSecret"];
        delete formattedAcitivity["spectateSecret"];
        delete formattedAcitivity["matchSecret"];
        delete formattedAcitivity["supportedPlatforms"];
        return (await this.client.request("SET_ACTIVITY", {
            pid: (pid ?? process) ? (process.pid ?? 0) : 0,
            activity: formattedAcitivity
        })).data;
    }
    /**
     * Used to clear a user's Rich Presence.
     *
     * @param pid - the application's process id
     */
    async clearActivity(pid) {
        await this.client.request("SET_ACTIVITY", { pid: (pid ?? process) ? (process.pid ?? 0) : 0 });
    }
    // #region Undocumented
    // This region holds method that are not documented by Discord BUT does exist
    // Also most of this might not even be correct, use at your own risk
    /**
     * Used to get a user's avatar
     * @param userId - id of the user to get the avatar of
     * @param format - image format
     * @param size - image size
     * @return base64 encoded image data
     */
    async getImage(userId, format = "png", size = 1024) {
        return (await this.client.request("GET_IMAGE", { type: "user", id: userId, format, size })).data.data_url;
    }
    /**
     * Requires RPC and RPC_VOICE_WRITE
     * @returns
     */
    async getSoundboardSounds() {
        return (await this.client.request("GET_SOUNDBOARD_SOUNDS")).data;
    }
    /**
     * Requires RPC and RPC_VOICE_WRITE
     * @returns
     */
    async playSoundboardSound(guildId, soundId) {
        return (await this.client.request("PLAY_SOUNDBOARD_SOUND", {
            guild_id: guildId,
            sound_id: soundId
        })).data;
    }
    /**
     * Requires RPC and RPC_VIDEO_WRITE
     * @returns
     */
    async toggleVideo() {
        return (await this.client.request("TOGGLE_VIDEO")).data;
    }
    /**
     * Requires RPC and RPC_SCREENSHARE_WRITE
     * @returns
     */
    async toggleScreenshare(pid) {
        return (await this.client.request("TOGGLE_SCREENSHARE", { pid })).data;
    }
    /**
     * Requires RPC and RPC_VOICE_WRITE
     * @returns
     */
    async setPushToTalk(active) {
        return (await this.client.request("PUSH_TO_TALK", { active })).data;
    }
    /**
     * Requires RPC and RPC_VOICE_WRITE
     * @returns
     */
    async setVoiceSettings(req) {
        return (await this.client.request("SET_VOICE_SETTINGS", req)).data;
    }
    /**
     * Requires RPC and RPC_VOICE_WRITE
     * @returns
     */
    async setVoiceSettings2(req) {
        return (await this.client.request("SET_VOICE_SETTINGS_2", req)).data;
    }
    /**
     * Requires RPC and RPC_GUILDS_MEMBERS_READ
     * @returns
     */
    async getChannelPermissions() {
        return (await this.client.request("GET_CHANNEL_PERMISSIONS")).data;
    }
    async getActivityInstanceConnectedParticipants() {
        return (await this.client.request("GET_ACTIVITY_INSTANCE_CONNECTED_PARTICIPANTS")).data;
    }
    async navigateToConnections() {
        return (await this.client.request("NAVIGATE_TO_CONNECTIONS")).data;
    }
    async createChanenlInvite(channelId, args) {
        return (await this.client.request("CREATE_CHANNEL_INVITE", { channel_id: channelId, ...args })).data;
    }
    async openExternalLink(url) {
        return (await this.client.request("OPEN_EXTERNAL_LINK", { url })).data;
    }
    async getPlatformBehaviors() {
        return (await this.client.request("GET_PLATFORM_BEHAVIORS")).data;
    }
    async getProviderAccessToken(provider, connectionRedirect) {
        return (await this.client.request("GET_PROVIDER_ACCESS_TOKEN", { provider, connectionRedirect })).data;
    }
    async maybeGetProviderAccessToken(provider) {
        return (await this.client.request("MAYBE_GET_PROVIDER_ACCESS_TOKEN", { provider })).data;
    }
    async getSKUS() {
        return (await this.client.request("GET_SKUS")).data;
    }
    async getEntitlements() {
        return (await this.client.request("GET_ENTITLEMENTS")).data;
    }
    async getSKUsEmbedded() {
        return (await this.client.request("GET_SKUS_EMBEDDED")).data;
    }
    async getEntitlementsEmbedded() {
        return (await this.client.request("GET_ENTITLEMENTS_EMBEDDED")).data;
    }
    async encourageHardwareAcceleration() {
        return (await this.client.request("ENCOURAGE_HW_ACCELERATION")).data;
    }
    async captureLog(level, message) {
        return (await this.client.request("CAPTURE_LOG", { level, message })).data;
    }
    async sendAnalyticsEvent(eventName, eventProperties) {
        return (await this.client.request("SEND_ANALYTICS_EVENT", { eventName, eventProperties })).data;
    }
    async getLocale() {
        return (await this.client.request("USER_SETTINGS_GET_LOCALE")).data.locale;
    }
    async getAchievements() {
        return (await this.client.request("GET_USER_ACHIEVEMENTS")).data;
    }
    async setAchievement(achievementId, percentComplete) {
        return (await this.client.request("SET_USER_ACHIEVEMENT", {
            achievement_id: achievementId,
            percent_complete: percentComplete
        })).data;
    }
    async createNetworkingToken() {
        return (await this.client.request("NETWORKING_CREATE_TOKEN")).data;
    }
    async networkingPeerMetrics() {
        return (await this.client.request("NETWORKING_PEER_METRICS")).data;
    }
    async networkingSystemMetrics() {
        return (await this.client.request("NETWORKING_SYSTEM_METRICS")).data;
    }
    async getNetworkingConfig() {
        return (await this.client.request("GET_NETWORKING_CONFIG")).data;
    }
    async startPurchase(skuId, pid) {
        return (await this.client.request("START_PURCHASE", { sku_id: skuId, pid })).data;
    }
    async startPremiumPurchase(pid) {
        return (await this.client.request("START_PREMIUM_PURCHASE", { pid })).data;
    }
    async getApplicationTicket() {
        return (await this.client.request("GET_APPLICATION_TICKET")).data;
    }
    async getEntitlementTicket() {
        return (await this.client.request("GET_ENTITLEMENT_TICKET")).data;
    }
    async validateApplication() {
        return (await this.client.request("VALIDATE_APPLICATION")).data;
    }
    async openOverlayVoiceSettings(pid) {
        return (await this.client.request("OPEN_OVERLAY_VOICE_SETTINGS", { pid })).data;
    }
    async openOverlayGuildInvite(code, pid) {
        return (await this.client.request("OPEN_OVERLAY_GUILD_INVITE", { code, pid })).data;
    }
    async openOverlayActivityInvite(type, pid) {
        const typeToNumber = {
            JOIN: 0
        };
        return (await this.client.request("OPEN_OVERLAY_ACTIVITY_INVITE", { type: typeToNumber[type], pid })).data;
    }
    async setOverlayLocked(locked, pid) {
        return (await this.client.request("SET_OVERLAY_LOCKED", { locked, pid })).data;
    }
    async browserHandoff() {
        return (await this.client.request("BROWSER_HANDOFF")).data;
    }
    async openGuildTemplateBrowser(code) {
        return (await this.client.request("GUILD_TEMPLATE_BROWSER", { code })).data;
    }
    async openGiftCodeBrowser(code) {
        return (await this.client.request("GIFT_CODE_BROWSER", { code })).data;
    }
    async brainTreePopupBridgeCallback(state, path, query) {
        return (await this.client.request("BRAINTREE_POPUP_BRIDGE_CALLBACK", { state, path, query })).data;
    }
    async billingPopupBridgeCallback(state, path, query, paymentSourceType) {
        return (await this.client.request("BILLING_POPUP_BRIDGE_CALLBACK", {
            state,
            path,
            query,
            payment_source_type: paymentSourceType
        })).data;
    }
    async connectionsCallback(providerType, code, openIdParams, state) {
        return (await this.client.request("CONNECTIONS_CALLBACK", {
            providerType: providerType,
            code,
            open_id_params: openIdParams,
            state
        })).data;
    }
    async deepLink(type, params) {
        return (await this.client.request("DEEP_LINK", { type, params })).data;
    }
    async inviteBrowser(code) {
        return (await this.client.request("INVITE_BROWSER", { code })).data;
    }
    async initiateImageUpload() {
        return (await this.client.request("INITIATE_IMAGE_UPLOAD")).data;
    }
    async openShareMomentDialog(mediaUrl) {
        return (await this.client.request("OPEN_SHARE_MOMENT_DIALOG", { mediaUrl })).data;
    }
    async openInviteDialog() {
        return (await this.client.request("OPEN_INVITE_DIALOG")).data;
    }
    async acceptActivityInvite(type, userId, sessionId, channelId, messageId) {
        const typeToNumber = {
            JOIN: 0
        };
        return (await this.client.request("ACCEPT_ACTIVITY_INVITE", {
            type: typeToNumber[type],
            user_id: userId,
            session_id: sessionId,
            channel_id: channelId,
            message_id: messageId
        })).data;
    }
    async activityInviteUser(userId, type, content, pid) {
        const typeToNumber = {
            JOIN: 0
        };
        return (await this.client.request("ACTIVITY_INVITE_USER", {
            user_id: userId,
            type: typeToNumber[type],
            content,
            pid
        })).data;
    }
    async closeActivityJoinRequest(userId) {
        return (await this.client.request("CLOSE_ACTIVITY_JOIN_REQUEST", { user_id: userId })).data;
    }
    async sendActivityJoinInvite(userId, pid) {
        return (await this.client.request("SEND_ACTIVITY_JOIN_INVITE", { user_id: userId, pid })).data;
    }
    async setConfig(useInteractivePip) {
        return (await this.client.request("SET_CONFIG", { use_interactive_pip: useInteractivePip })).data;
    }
}
exports.ClientUser = ClientUser;
